import 'package:digi_bank/src/app/routes/routes_handler.dart';
import 'package:digi_bank/src/common/services/settings_service.dart';
import 'package:digi_bank/src/presentation/screens/home/view/home_screen.dart';
import 'package:digi_bank/src/presentation/screens/reward/view/reward_screen.dart';
import 'package:digi_bank/src/presentation/screens/setting/view/setting_screen.dart';
import 'package:digi_bank/src/presentation/screens/virtual_card/view/virtual_card_screen.dart';
import 'package:digi_bank/src/presentation/screens/wallet/view/wallet_screen.dart';
import 'package:flutter/foundation.dart';
import 'package:flutter/material.dart';
import 'package:get/get.dart';

class NavigationController extends GetxController {
  final RxInt selectedIndex = 0.obs;
  late List<List<Widget>> pageStacks;
  final Map<String, dynamic> _routeArguments = {};
  final RxString multipleCurrency = "".obs;
  final RxString virtualCard = "".obs;
  final RxString userReward = "".obs;
  final RxBool isLoading = false.obs;

  @override
  void onInit() {
    super.onInit();
    loadData();
  }

  Future<void> loadData() async {
    await loadSettings();
    pageStacks = [
      [HomeScreen()],
      if (virtualCard.value != "0") [VirtualCardScreen()],
      if (multipleCurrency.value != "0") [WalletScreen()],
      if (userReward.value != "0") [RewardScreen()],
      [SettingScreen()],
    ];
  }

  Future<void> loadSettings() async {
    isLoading.value = true;
    final multipleCurrencyValue = await SettingsService.getSettingValue(
      'multiple_currency',
    );
    multipleCurrency.value = multipleCurrencyValue ?? "";
    final virtualCardValue = await SettingsService.getSettingValue(
      'virtual_card',
    );
    virtualCard.value = virtualCardValue ?? "";
    final userRewardValue = await SettingsService.getSettingValue(
      'user_reward',
    );
    userReward.value = userRewardValue ?? "";
    isLoading.value = false;
  }

  void onTapItem(int index) {
    if (selectedIndex.value == index && pageStacks[index].length > 1) {
      pageStacks[index] = [pageStacks[index].first];
      update();
    }
    selectedIndex.value = index;
  }

  void pushPage(Widget page) {
    pageStacks[selectedIndex.value].add(page);
    update();
  }

  void pushNamed(String routeName, {dynamic arguments}) {
    if (kDebugMode) {
      print('Pushing route: $routeName with arguments: $arguments');
    }

    final route = _findRoute(routeName);
    if (route != null) {
      if (arguments != null) {
        _routeArguments[routeName] = arguments;
      }

      final widget = route.page();
      pageStacks[selectedIndex.value].add(widget);
      update();
    } else {
      throw Exception('Route $routeName not found');
    }
  }

  void pushOffNamed(String routeName, {dynamic arguments}) {
    if (kDebugMode) {
      print('Push off route: $routeName with arguments: $arguments');
    }

    final route = _findRoute(routeName);
    if (route != null) {
      if (arguments != null) {
        _routeArguments[routeName] = arguments;
      }

      if (pageStacks[selectedIndex.value].isNotEmpty) {
        pageStacks[selectedIndex.value].removeLast();
      }
      final widget = route.page();
      pageStacks[selectedIndex.value].add(widget);
      update();
    } else {
      throw Exception('Route $routeName not found');
    }
  }

  void pushOffAllNamed(String routeName, {dynamic arguments}) {
    if (kDebugMode) {
      print('Push off all route: $routeName with arguments: $arguments');
    }

    final route = _findRoute(routeName);
    if (route != null) {
      if (arguments != null) {
        _routeArguments[routeName] = arguments;
      }

      final widget = route.page();
      pageStacks[selectedIndex.value] = [widget];
      update();
    } else {
      throw Exception('Route $routeName not found');
    }
  }

  dynamic getArguments(String routeName) {
    return _routeArguments[routeName];
  }

  GetPage? _findRoute(String routeName) {
    if (kDebugMode) {
      print('GOING TO ROUTE: $routeName');
    }
    try {
      return routesHandler.firstWhere((route) => route.name == routeName);
    } catch (e) {
      if (kDebugMode) {
        print('Route not found: $routeName');
      }
      return null;
    }
  }

  bool popPage() {
    if (pageStacks[selectedIndex.value].length > 1) {
      pageStacks[selectedIndex.value].removeLast();
      update();
      return true;
    }
    return false;
  }

  Widget get currentPage => pageStacks[selectedIndex.value].last;
}
